; -------------------------------------------------------------
; Copyright 2024 University of Calgary
;
; Licensed under the Apache License, Version 2.0 (the "License");
; you may not use this file except in compliance with the License.
; You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.
; -------------------------------------------------------------

pro __aurorax_ucalgary_readfile_grid, $
  filename, $
  data, $
  timestamp_list, $
  meta, $
  start_dt = start_dt, $
  end_dt = end_dt, $
  first_frame = first_frame, $
  verbose = verbose
  compile_opt hidden

  if not isa(verbose) then verbose = 1

  ; Convert scalar filename to length 1 array so we can 'iterate' regardless
  if isa(filename, /scalar) then filename = [filename]

  ; If start_dt or end_dt were passed, we need to cut down the filenames accordingly
  if keyword_set(start_dt) or keyword_set(end_dt) then begin
    if keyword_set(start_dt) then begin
      start_yy = strmid(start_dt, 0, 4)
      start_mm = strmid(start_dt, 5, 2)
      start_dd = strmid(start_dt, 8, 2)
      start_hr = strmid(start_dt, 11, 2)
      start_mn = strmid(start_dt, 14, 2)
    endif
    if keyword_set(end_dt) then begin
      end_yy = strmid(end_dt, 0, 4)
      end_mm = strmid(end_dt, 5, 2)
      end_dd = strmid(end_dt, 8, 2)
      end_hr = strmid(end_dt, 11, 2)
      end_mn = strmid(end_dt, 14, 2)
    endif

    hr_mn = []
    foreach f, filename do begin
      if n_elements(strsplit(f, start_yy + start_mm + start_dd + '_', /extract, /regex)) eq 1 then begin
        hr_mn = [hr_mn, 'nan']
      endif else begin
        hr_mn = [hr_mn, strmid((strsplit(f, start_yy + start_mm + start_dd + '_', /extract, /regex))[-1], 0, 4)]
      endelse
    endforeach

    start_dt_idx = where(hr_mn eq start_hr + start_mn, /null)
    end_dt_idx = where(hr_mn eq end_hr + end_mn, /null)

    ; Check that the start/end time range actually corresponds to the files passed in
    if start_dt_idx eq !null then begin
      print, '[aurorax_read] Error - start_dt does not correspond to any of the input filenames'
      return
    endif else if end_dt_idx eq !null then begin
      print, '[aurorax_read] Error - end_dt does not correspond to any of the input filenames'
      return
    endif

    ; if everything worked properly we can now slice out the filenames we actually want to read
    filename = filename[start_dt_idx : end_dt_idx]
  end
  filename = filename.toarray()

  n_files = n_elements(filename)
  if (n_files gt 1) then filename = filename[sort(filename)]

  ; Setting up master lists to hold data for multi-file reading
  master_timestamp = []
  master_file_meta = []
  master_frame_meta = []
  frames_read_counter = 0ul
  break_after_first = 0

  foreach f, filename, file_num do begin
    if (verbose gt 0) then print, '[aurorax_read] Reading file: ' + f

    ; Reading the grid data into an
    ; array object to be returned.
    file_id = h5f_open(f)

    data_group_id = h5g_open(file_id, 'data')
    grid_dataset_id = h5d_open(data_group_id, 'grid')
    source_info_group_id = h5g_open(data_group_id, 'source_info')
    confidence_dataset_id = h5d_open(source_info_group_id, 'confidence')
    grid = h5d_read(grid_dataset_id)
    confidence = h5d_read(confidence_dataset_id)

    ; Transposing the grids for proper IDL dimensionality
    if size(grid, /n_dimensions) eq 3 then begin
      grid = reverse(transpose(grid, [1, 2, 0]), 2)
      confidence = reverse(transpose(confidence, [1, 2, 0]), 2)
    endif else if size(grid, /n_dimensions) eq 4 then begin
      grid = reverse(transpose(grid, [1, 2, 3, 0]), 3)
      confidence = reverse(transpose(confidence, [1, 2, 0]), 2)
    endif

    if n_elements(size(grid, /dimensions)) eq 2 then grid = reform(grid, [size(grid, /dimensions), 1])

    grid_dims = size(grid, /dimensions)
    conf_dims = size(confidence, /dimensions)

    ; if this is the first file read in, allocate memory for the data
    if file_num eq 0 then begin
      ; allocate memory for grid data assuming 20 frames per file (will be trimmed at the end)
      predicted_n_frames = 20 * n_files
      master_grid = make_array([grid_dims[0 : n_elements(grid_dims) - 2], predicted_n_frames], type = size(grid, /type))
      master_confidence = make_array([conf_dims[0 : n_elements(conf_dims) - 2], predicted_n_frames], type = size(conf_dims, /type))

      ; insert the first file's data into the newly allocated arrays
      n_frames = (size(grid, /dimensions))[-1]

      if n_elements(grid_dims) eq 3 then begin
        if keyword_set(first_frame) then begin
          n_frames = 1
          master_grid[*, *, frames_read_counter : frames_read_counter] = grid[*, *, 0]
          master_confidence[*, *, frames_read_counter : frames_read_counter] = confidence[*, *, 0]
        endif else begin
          master_grid[*, *, frames_read_counter : frames_read_counter + n_frames - 1] = grid
          master_confidence[*, *, frames_read_counter : frames_read_counter + n_frames - 1] = confidence
        endelse
      endif else if n_elements(grid_dims) eq 4 then begin
        if keyword_set(first_frame) then begin
          n_frames = 1
          master_grid[*, *, *, frames_read_counter : frames_read_counter] = grid[*, *, *, 0]
          master_confidence[*, *, frames_read_counter : frames_read_counter] = confidence[*, *, 0]
        endif else begin
          master_grid[*, *, *, frames_read_counter : frames_read_counter + n_frames - 1] = grid
          master_confidence[*, *, frames_read_counter : frames_read_counter + n_frames - 1] = confidence
        endelse
      endif else begin
        print, '[aurorax_read] Error - Currently only 2 or 3 dimensional grids can be read in, found n_dims = ' + strcompress(string(n_elements(grid_dims)), /remove_all)
      endelse
    endif else begin
      if keyword_set(first_frame) then begin
        ; insert this file's data into the arrays
        n_frames = 1
        if n_elements(grid_dims) eq 3 then begin
          master_grid[*, *, frames_read_counter : frames_read_counter + n_frames - 1] = grid[*, *, 0]
          master_confidence[*, *, frames_read_counter : frames_read_counter + n_frames - 1] = confidence[*, *, 0]
        endif else if n_elements(grid_dims) eq 4 then begin
          master_grid[*, *, *, frames_read_counter : frames_read_counter + n_frames - 1] = grid[*, *, *, 0]
          master_confidence[*, *, frames_read_counter : frames_read_counter + n_frames - 1] = confidence[*, *, 0]
        endif else begin
          print, '[aurorax_read] Error - Currently only 2 or 3 dimensional grids can be read in, found n_dims = ' + strcompress(string(n_elements(grid_dims)), /remove_all)
        endelse
      endif else begin
        ; insert this file's data into the arrays
        n_frames = (size(grid, /dimensions))[-1]
        if n_elements(grid_dims) eq 3 then begin
          master_grid[*, *, frames_read_counter : frames_read_counter + n_frames - 1] = grid
          master_confidence[*, *, frames_read_counter : frames_read_counter + n_frames - 1] = confidence
        endif else if n_elements(grid_dims) eq 4 then begin
          master_grid[*, *, *, frames_read_counter : frames_read_counter + n_frames - 1] = grid
          master_confidence[*, *, frames_read_counter : frames_read_counter + n_frames - 1] = confidence
        endif else begin
          print, '[aurorax_read] Error - Currently only 2 or 3 dimensional grids can be read in, found n_dims = ' + strcompress(string(n_elements(grid_dims)), /remove_all)
        endelse
      endelse
    endelse

    ; update the number of frames we've read in
    frames_read_counter += n_frames

    ; The returned 'meta' variable will be an IDL structure that contains the
    ; file level metadata, the frame level metadata, as well as the 'timestamp'
    ; array. So we read each of those into seperate objects, and then create
    ; the struct from there

    ; Reading in the timestamp to an array object to be returned
    timestamp_dataset_id = h5d_open(data_group_id, 'timestamp')
    timestamp = h5d_read(timestamp_dataset_id)

    ; Reading in the file level metadata into a hash and then converting to IDL struct
    meta_group_id = h5g_open(file_id, 'metadata')
    file_meta_dataset_id = h5d_open(meta_group_id, 'file')
    n_file_meta_attributes = h5a_get_num_attrs(file_meta_dataset_id)

    ; Iterating through each attribute and adding to hash, then converting to struct
    file_meta_hash = hash()
    for i = 0, (n_file_meta_attributes - 1) do begin
      attribute_id = h5a_open_idx(file_meta_dataset_id, i)
      attribute_name = h5a_get_name(attribute_id)
      attribute = h5a_read(attribute_id)
      file_meta_hash[attribute_name] = attribute
    endfor

    if ~isa(file_meta) then file_meta = file_meta_hash.toStruct()

    ; Reading in the frame level metadata into a list, where each list element is a meta
    ; structure belonging to the frame of that element's list index
    frame_meta_group_id = h5g_open(meta_group_id, 'frame')
    n_frame_meta_datasets = h5g_get_nmembers(meta_group_id, 'frame')

    ; Iterating through each frame dataset in the frame meta group
    for i = 0, (n_frame_meta_datasets - 1) do begin
      frame_meta_dataset_id = h5d_open(frame_meta_group_id, 'frame' + strcompress(string(i), /remove_all))
      n_frame_meta_attributes = h5a_get_num_attrs(frame_meta_dataset_id)

      ; Iterating through each attribute for the current iteration's dataset and adding to hash
      frame_meta_hash = hash()
      for j = 0, (n_frame_meta_attributes - 1) do begin
        attribute_id = h5a_open_idx(frame_meta_dataset_id, j)
        attribute_name = h5a_get_name(attribute_id)
        attribute = h5a_read(attribute_id)
        frame_meta_hash[attribute_name] = attribute
      endfor

      ; Converting hash to struct and then appending to frame meta list
      frame_meta_struct = frame_meta_hash.toStruct()
      master_frame_meta = [master_frame_meta, frame_meta_struct]

      ; break out of metadata reading after one frame if /first_frame is set
      if keyword_set(first_frame) then break
    endfor

    ; If first_frame_only keyword is set, slice timestamp accordingly
    if keyword_set(first_frame) then timestamp = timestamp[0]

    ; Append to master timestamp
    master_timestamp = [master_timestamp, timestamp]

    h5_close
  endforeach

  ; Removing frames with no data
  master_grid_dims = size(master_grid, /dimensions)
  master_conf_dims = size(master_confidence, /dimensions)
  if n_elements(master_grid_dims) eq 3 then master_grid = master_grid[*, *, 0 : frames_read_counter - 1]
  if n_elements(master_grid_dims) eq 4 then master_grid = master_grid[*, *, *, 0 : frames_read_counter - 1]
  if n_elements(master_conf_dims) eq 3 then master_confidence = master_confidence[*, *, 0 : frames_read_counter - 1]
  timestamp_list = master_timestamp

  ; Create metadata structure to return
  meta = {timestamp: master_timestamp, file_meta: file_meta, frame_meta: master_frame_meta}

  ; Create data structure to return
  data = {grid: master_grid, confidence: master_confidence}

  first_frame_jump:
end
