; -------------------------------------------------------------
; Copyright 2024 University of Calgary
;
; Licensed under the Apache License, Version 2.0 (the "License");
; you may not use this file except in compliance with the License.
; You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.
; -------------------------------------------------------------

;+
; :Description:
;       Initialize ATM 'inverse' calculation output flag settings.
;
;       Create a hash object which is used to represent all output values included in
;       an ATM inverse calculation. ATM calculations are performed in a way where
;       you can toggle ON/OFF whichever pieces of information you do or don't want.
;       This improves the efficiency of the calculation routine resulting in faster queries.
;
;       By default, all output flags are disabled. There exist several helper keywords
;       to enable all flags, or enable only height-integrated Rayleighs flags.
;
; :Keywords:
;       set_all_true: in, optional, Boolean
;         Enable all output flags.
;
; :Returns:
;       Hash
;
; :Examples:
;       aurorax_atm_inverse_get_output_flags()
;       aurorax_atm_inverse_get_output_flags(/SET_ALL_TRUE)
;+
function aurorax_atm_inverse_get_output_flags, set_all_true = set_all_true
  ; create hash
  output_flags = hash()
  output_flags['altitudes'] = 0
  output_flags['energy_flux'] = 0
  output_flags['mean_energy'] = 0
  output_flags['oxygen_correction_factor'] = 0

  ; set all true, if necessary
  if keyword_set(set_all_true) then begin
    foreach value, output_flags, key do begin
      output_flags[key] = 1
    endforeach
  endif

  ; return
  return, output_flags
end
